import path from 'node:path';
import os from 'node:os';
import fs, { writeFileSync } from 'node:fs';
import { Buffer } from 'node:buffer';
import DigestFetch from 'digest-fetch';
import chalk from 'chalk';
import { Config, SearchResult, User, UserInfoSearch } from './types.ts';

const getWorkDirectory = () => {

  const workDirectoryPath = path.join(os.homedir(), 'Documents', 'netgym', 'hikvision');

  if (!fs.existsSync(workDirectoryPath)) 
    fs.mkdirSync(workDirectoryPath, { recursive: true });

  return workDirectoryPath;

}

const loadConfig = (): Config => {

  try {

    // Get config file path
    const configFilePath = path.join(getWorkDirectory(), 'config.json');

    // Get config file content
    const fileContent = fs.readFileSync(configFilePath, 'utf8');

    // return config
    return JSON.parse(fileContent);

  }
  catch(error) {
    console.error('Error in credentials');
    throw new Error(error);
  }

}

const fetchUsersInfo = async (page: number): Promise<UserInfoSearch> => {

  try {

    const response = await digestClient.fetch(
      `http://${config.ip}/ISAPI/AccessControl/UserInfo/Search?format=json`, 
      {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ "UserInfoSearchCond": { "searchId": "0", "searchResultPosition": page * 30, "MaxResults": 30 } })
      }
    );

    const result: SearchResult = await response.json();
    return result.UserInfoSearch;

  } catch (error) {
    console.error('Error:', error);
    throw new Error("Error fetching data");
  }
}

const getUsers = async (): Promise<User[]> => {

  console.log(chalk.blue('Importing users...\n'));

  let responseStatusStrg: string;

  let currentPage = 0;

  const users: User[] = [];

  do {

    // Get Users Info
    const searchInfo = await fetchUsersInfo(currentPage);

    // Push new users to users array
    searchInfo.UserInfo.forEach(user => { users.push({ 'id': user.employeeNo, 'name': user.name, 'imageUrl': user.faceURL }) })

    // Increase page
    currentPage++;

    // Set new stauts response
    responseStatusStrg = searchInfo.responseStatusStrg;

  } while (responseStatusStrg === 'MORE');

  return users.sort((a, b) => a.id.localeCompare(b.id));

}

const importUsersImages = async (users: User[]) => {

  // Create images directory if not exists
  const imagesDirectoryPath = path.join(getWorkDirectory(), 'images');
  if (!fs.existsSync(imagesDirectoryPath)) fs.mkdirSync(imagesDirectoryPath, { recursive: true });

  console.log(chalk.blueBright('Importing images...\n'));

  for (const user of users) {

    if (!user.imageUrl) {
      console.log(chalk.yellow(`[!!] (${user.id}) ${user.name} has not image`));
      continue;
    }

    try {
      // Get image
      const response = await digestClient.fetch(user.imageUrl);

      // Get buffer
      const arrayBuffer = await response.arrayBuffer();
      const buffer = Buffer.from(arrayBuffer);

      // Save image
      writeFileSync(path.join(imagesDirectoryPath, `${user.id}.jpg`), buffer);

      console.log(`[OK] (${user.id}) ${user.name} image stored`);
    }
    catch(error) { console.log(chalk.red(`[!!] Error storing (${user.id}) ${user.name} image`)); }

  }

}

const importUsersInformation = async () => {

  // Get users information
  const users = await getUsers();
  console.log(`${users.length} customers found\n`);

  // Define users JSON file path in user/documents/netgym/hikvision/users.json
  const usersFilePath = path.join(getWorkDirectory(), 'users.json');

  // Store users in a JSON file
  fs.writeFileSync(usersFilePath, JSON.stringify(users), 'utf8');
  console.log(chalk.greenBright(`Users imported at: ${usersFilePath}\n`));

  // Store users images
  await importUsersImages(users);
  console.log(chalk.greenBright(`\nUsers images imported at: ${path.join(getWorkDirectory(), 'images')}\n`));

  console.log(chalk.greenBright('Data importing finished!'));
}

const config = loadConfig();
const digestClient = new DigestFetch(config.user, config.password);
importUsersInformation();